// --------------------- Vaunix LMS Example Program -----------------
//
// This program shows how to use the Vaunix LMS Library
// 
// ------------------------------------------------------------------
#include <stdbool.h>
#include <stdio.h>
#include <unistd.h>
#include <math.h>
#include "LMShid.h"

#define FALSE 0
#define TRUE !FALSE

// convert from our integer 10Hz representation to a floating point value in GHz
float FreqInGHz(int frequency10Hz) {

    return (float)frequency10Hz / 1.0e8;
}




int main (int argc, char *argv[]) {
  int nDevices, nActive;
  int i, result, status;
  int midpoint;
  int powerlevel;
  float swp_start, swp_end;
  float fresult;
  char cModelName[32];
  char c;
  char *username;
  DEVID activeDevices[MAXDEVICES];

  username = getlogin();
  if (0 != strcmp(username, "root")) {
    printf("Hi %s,\r\n", username);
    printf("Accessing USB ports on a Linux machine require root level\r\n");
    printf("access. You are not logged in as root. You may be able to\r\n");
    printf("proceed if you have used 'chmod' to change the access mode\r\n");
    printf("of the appropriate devices in /dev/bus/usb. That requires\r\n");
    printf("root access also. We'll continue, but if you don't see your\r\n");
    printf("LMS devices or no data can be read from or written to them,\r\n");
    printf("that's probably the problem. su to root and try again.\r\n\r\n");
    printf("If you ran 'su' and became root before running this, you're fine.\r\n\r\n");
  }
  fnLMS_Init();
  fnLMS_SetTestMode(FALSE);
  nDevices = fnLMS_GetNumDevices();
  printf("LMS test/demonstration program using library version %s\r\n\r\n", fnLMS_LibVersion());
  if (0 == nDevices) {
    printf("No Vaunix LMS devices located. Would you like to run in test mode? "); fflush(0);
    c = getchar();
    if ('Y' == (c & 0xdf)) {
      printf("\r\nSwitching to test mode.\r\n");
      fnLMS_Init();
      fnLMS_SetTestMode(TRUE);
      nDevices = fnLMS_GetNumDevices();
    }
  }
  printf("Found %d devices\r\n", nDevices);

  for (i=1; i<=nDevices; i++) {
    result = fnLMS_GetModelName(i, cModelName);
    printf("  Model %d is %s (%d chars)\r\n", i, cModelName, result);
  }
  printf("\r\n");
  
  nActive = fnLMS_GetDevInfo(activeDevices);
  printf("We have %d active devices\r\n", nActive);

  for (i=0; i<nActive; i++) {
    if (i > 0) printf("\r\n");
    printf("  Device %d is active\r\n", activeDevices[i]);
    status = fnLMS_InitDevice(activeDevices[i]);
    printf("  Opened device %d. Return status=0x%08x (%s)\r\n", activeDevices[i], status, fnLMS_perror(status));

    printf("Before we do any tests, let's let the DLL get to know the device...\r\n"); fflush(0);
    sleep(1);
   
    // --- show the state of the device ---
    status = fnLMS_GetModelName(activeDevices[i], cModelName);
    printf("  Device %d (%s) has ", activeDevices[i], cModelName);
    status = fnLMS_GetSerialNumber(activeDevices[i]);
    printf("serial number = %d\r\n", status);

    result = fnLMS_GetFrequency(activeDevices[i]);
    if (result & ERROR_STATUS_MSK) printf(" ERROR: Status=0x%08x (%s) in fnLMS_GetFrequency\r\n", result, fnLMS_perror(result));
    else printf("  Frequency =  %f GHz\r\n", FreqInGHz(result));

    result = fnLMS_GetStartFrequency(activeDevices[i]);
    if (result & ERROR_STATUS_MSK) printf(" ERROR: Status=0x%08x (%s) in fnLMS_GetStartFrequency\r\n", result, fnLMS_perror(result));
    else printf("  Sweep start frequency =  %f GHz\r\n", FreqInGHz(result));

    result = fnLMS_GetEndFrequency(activeDevices[i]);
    if (result & ERROR_STATUS_MSK) printf(" ERROR: Status=0x%08x (%s) in fnLMS_GetEndFrequency\r\n", result, fnLMS_perror(result));
    else printf("  Sweep end frequency = %f GHz\r\n", FreqInGHz(result));

    result = fnLMS_GetSweepTime(activeDevices[i]);
    if (result & ERROR_STATUS_MSK) printf(" ERROR: Status=0x%08x (%s) in fnLMS_GetSweepTime\r\n", result, fnLMS_perror(result));
    else printf("  Sweep time =  %f seconds\r\n", ((float)result * .001));

    result = fnLMS_GetPowerLevel(activeDevices[i]);
    // testing the result for an error is a bit more complicated since the power level can be a negative value
    if (result == BAD_PARAMETER || result == BAD_HID_IO || result == DEVICE_NOT_READY)
         printf(" ERROR: Status=0x%08x (%s) in fnLMS_GetPowerLevel\r\n", result, fnLMS_perror(result));
    else printf("  Power level = %f db\r\n", ((float)result / 4));

    fresult = fnLMS_GetPulseOnTime(activeDevices[i]);
    if (fresult < 0) printf(" ERROR: Status= %f (%s) in fnLMS_GetPulseOnTime\r\n", fresult, fnLMS_pFloatError(fresult));
    else printf("  Pulse on time = %f sec.\r\n", fresult);

    fresult = fnLMS_GetPulseOffTime(activeDevices[i]);
    if (fresult < 0) printf(" ERROR: Status= %f (%s) in fnLMS_GetPulseOffTime\r\n", fresult, fnLMS_pFloatError(fresult));
    else printf("  Pulse off time = %f sec.\r\n", fresult);

    // --- set some (arbitrary) values based on the range of the Lab Brick ---
    result = fnLMS_GetMinFreq(activeDevices[i]);
    if (result & ERROR_STATUS_MSK) printf(" ERROR: Status=0x%08x (%s) in fnLMS_GetMinFrequency\r\n", result, fnLMS_perror(result));
    else printf("  Minimum Frequency =  %f GHz\r\n", FreqInGHz(result));  
    
    result = fnLMS_GetMaxFreq(activeDevices[i]);
    if (result & ERROR_STATUS_MSK) printf(" ERROR: Status=0x%08x (%s) in fnLMS_GetMaxFrequency\r\n", result, fnLMS_perror(result));
    else printf("  Maximum Frequency =  %f GHz\r\n", FreqInGHz(result));  

    // --- find the offset to the midpoint in the Lab Brick's frequency range ---
    midpoint = (fnLMS_GetMaxFreq(activeDevices[i]) - fnLMS_GetMinFreq(activeDevices[i])) / 2;
  
    // --- set up a sweep ---
    swp_start = (int) ((float)midpoint * .33) + fnLMS_GetMinFreq(activeDevices[i]);	// start the sweep at 1/6 of the overall bandwidth
    swp_end = (int) ((float)midpoint * .66) + fnLMS_GetMinFreq(activeDevices[i]);	// end the sweep at 1/3 of the overall bandwidth

    // --- find the frequency in the middle of the Lab Brick's range ---
    midpoint = midpoint + fnLMS_GetMinFreq(activeDevices[i]);

    status = fnLMS_SetFrequency(activeDevices[i],  midpoint);
    if (result & ERROR_STATUS_MSK) printf(" ERROR: Status=0x%08x (%s) in fnLMS_SetFrequency\r\n", status, fnLMS_perror(status));
    else printf("  Set frequency to %f GHz. \r\n", FreqInGHz(midpoint));

    result = fnLMS_GetFrequency(activeDevices[i]);
    printf("  Read frequency back and got %d in 10Hz units\r\n", result);

    status = fnLMS_SetStartFrequency(activeDevices[i], swp_start);
    if (result & ERROR_STATUS_MSK) printf(" ERROR: Status=0x%08x (%s) in fnLMS_SetStartFrequency\r\n", status, fnLMS_perror(status));
    else printf("  Set sweep start frequency to %f GHz.\r\n", FreqInGHz(swp_start) );
    
    status = fnLMS_SetEndFrequency(activeDevices[i], swp_end);
    if (result & ERROR_STATUS_MSK) printf(" ERROR: Status=0x%08x (%s) in fnLMS_SetEndFrequency\r\n", status, fnLMS_perror(status));
    else printf("  Set sweep end frequency to %f GHz.\r\n", FreqInGHz(swp_end) );
    
    result = fnLMS_GetStartFrequency(activeDevices[i]);
    printf("  Read start frequency back and got %d in 10Hz units\r\n", result);
    result = fnLMS_GetEndFrequency(activeDevices[i]);
    printf("  Read end frequency back and got %d in 10Hz units\r\n", result);

    status = fnLMS_SetSweepTime(activeDevices[i], 20);
    printf("  Set sweep time to 20 milliseconds. Status=0x%08x (%s)\r\n", status, fnLMS_perror(status));    
    result = fnLMS_GetSweepTime(activeDevices[i]);
    printf("  Read sweep time =  %f seconds\r\n", ((float)result * .001));

    // set the powerlevel to -15db, using our .25db units
    powerlevel = -15 * 4;	
    status = fnLMS_SetPowerLevel(activeDevices[i], powerlevel);
    printf("  Set power level to -15 db. Status=0x%08x (%s)\r\n", status, fnLMS_perror(status));    
    result = fnLMS_GetPowerLevel(activeDevices[i]);
    printf("  Read power level = %f db\r\n", ((float)result / 4));

    status = fnLMS_SetRFOn(activeDevices[i], TRUE);
    printf("  Set RF on. Status=0x%08x (%s)\r\n", status, fnLMS_perror(status));    

    // --- turn on pulse modulation ---
    status = fnLMS_SetUseExternalPulseMod(activeDevices[i], FALSE);
    printf("  SetUseExternalPulseMod is FALSE. Status=0x%08x (%s)\r\n", status, fnLMS_perror(status));    

    status = fnLMS_SetFastPulsedOutput(activeDevices[i], .002, .05, TRUE);
    printf("  SetFastPulsedOutput to 2ms on every 50ms, Status=0x%08x (%s)\r\n", status, fnLMS_perror(status));    


    status = fnLMS_SetUseInternalRef(activeDevices[i], TRUE);
    printf("  Set UseInternalRef TRUE. Status=0x%08x (%s)\r\n", status, fnLMS_perror(status));    

    status = fnLMS_SetSweepDirection(activeDevices[i], TRUE);
    printf("  Set sweep direction TRUE. Status=0x%08x (%s)\r\n", status, fnLMS_perror(status));    

    status = fnLMS_SetSweepMode(activeDevices[i], TRUE);
    printf("  Set sweep mode TRUE. Status=0x%08x (%s)\r\n", status, fnLMS_perror(status));    

    status = fnLMS_SetSweepType(activeDevices[i], TRUE);
    printf("  Set sweep type TRUE. Status=0x%08x (%s)\r\n", status, fnLMS_perror(status));    

    status = fnLMS_StartSweep(activeDevices[i], TRUE);
    printf("  Started sweep. Status=0x%08x (%s)\r\n", status, fnLMS_perror(status));    

    status = fnLMS_CloseDevice(activeDevices[i]);
    printf("  Closed device %d. Return status=0x%08x (%s)\r\n", activeDevices[i], status, fnLMS_perror(status));
  }
  printf("End of test\r\n");
  return 0;
}
